<?php

namespace App\Services\Video;

use App\Contracts\Video\VideoActionService;
use App\Services\BaseServiceImpl;

use Carbon\Carbon;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

use App\Models\Contact;
use App\Models\EmailOnQueue;
use App\Models\Message;
use App\Models\Permission;
use App\Models\Twilio;
use App\Models\VideoRecording;
use App\Models\VideoRoom;
use App\Models\VideoRoomParticipant;

use Twilio\Jwt\AccessToken;
use Twilio\Jwt\Grants\VideoGrant;
use Twilio\Rest\Client;

class VideoActionServiceImpl extends BaseServiceImpl implements VideoActionService
{

    protected $sid;
    protected $token;
    protected $key;
    protected $secret;
    protected $twilio;

    public function __construct()
    {
        $this->sid = config('services.twilio.sid');
        $this->token = config('services.twilio.token');
        $this->key = config('services.twilio.key');
        $this->secret = config('services.twilio.secret');
        $this->maxParticipants = config('services.twilio.maxParticipants');
        $this->twilio = new Twilio;
    }

    public function getVideos()
    {
        $access = session('allUserAccesses')['videos'];
        $videos = VideoRoom::all();
        $result = array();
        foreach ($videos as $video)
        {
            $view = ' <a class="btn btn-primary btn-sm" href="/videos/room/join/'.$video->room_name.'/'.$video->room_id.'">
                            <i class="fa fa-video-camera">
                            </i>
                            Join
                        </a>';
            
            if($video->status==VideoRoom::VIDEO_COMPLETED){
                $status = '<span class="badge badge-success">Completed</span>';
                $view = "<i class='fa fa-lock' title=''></i>".' <span class="badge badge-danger">Room already closed</span>';
            } else {
                $status = '<span class="badge badge-warning">In Progress</span>';
            }
            $file = "";
            if (Permission::canDownload($access))
            {
                $recordings = VideoRecording::where('video_room_id', $video->id)->get();
                foreach ($recordings as $recording)
                {
                    if($recording->status == VideoRoom::VIDEO_COMPLETED)
                    {
                        $file .= "<a href='/videos/{$recording->sid}/downloadRecording'>
                                    <i class='fa fa-file-video-o' title='{$recording->sid}.mp4'></i>
                                </a>";    
                    }
                }
            } else {
                $file = "<i class='fa fa-exclamation-circle' title=''></i>".' <span class="badge badge-danger">Access not available</span>';
            }

            $result[] = array(
                $video->id,
                //$video->room_name,
                $video->room_id,
                $video->mobile_number,
                $video->created_at->format('Y-m-d H:i:s'),
                $status,
                $file,
                $view
            );
        }

        $resultObject = (object) [
            'videos' => $result
        ];

        return $resultObject;
    }

    public function createRoom($request)
    {
        if (isset($request->mobileNumber))
        {
            $mobileNumber = $request->mobileNumber;
            $roomName = $request->mobileNumber . date('YmdHis');
            $emailAddress="";
        } else {
            $roomName = time() . date('YmdHis');    
            $mobileNumber = "4845748747";
            $emailAddress = $request->emailAddress;
        }
        if (!$this->twilio->isRoomExists($roomName)) {
            $room = $this->twilio->createRoom($roomName);
            if (isset($request->mobileNumber))
            {
                $parameter = array(
                    'mobileNumber' => $request->mobileNumber,
                    'message'   => "Here's the link for video connection " .url('/').'/videos/room/join/'.$roomName.'/'.$room->sid,
                );
                $this->sendSMS($parameter);
            } else {
                $data = array(
                    'link' => url('/') . '/videos/room/join/' . $roomName.'/'.$room->sid,
                );
            
                $emailParameters = array(
                    'subject' => 'Stratvideo Onboarding Meeting Link',
                    'template' => "mails.pcr",
                    'parameters' => serialize($data),
                    'emailAddress' => $emailAddress,
                );

                $this->sendEmail($emailParameters);
        
            }
            
            $video = new VideoRoom;
            $video->mobile_number = $mobileNumber;
            $video->email_address = $emailAddress;
            $video->room_id = $room->sid;
            $video->room_name = $roomName;
            $video->link = url('/') . '/videos/room/join/' . $roomName.'/'.$room->sid;
            $video->status = $room->status;
            $video->user_id = auth()->user()->id;

            //additional optional fields
            if(isset($request->shipped_date))
            {
                $video->shipped_date = $request->shipped_date;    
            }
            if(isset($request->shipped_time))
            {
                $video->shipped_time = $request->shipped_time;    
            }
            if(isset($request->shipped_by))
            {
                $video->shipped_by = $request->shipped_by;    
            }
            if(isset($request->shipping_label_company))
            {
                $video->shipping_label_company = $request->shipping_label_company;    
            }
            if(isset($request->shipping_company))
            {
                $video->shipping_company = $request->shipping_company;    
            }
            $video->save();
            Cache::put($room->sid, $room->status);
            \Log::debug("created new room: " . $roomName);
        
            $resultObject = (object) [
                'roomName' => $roomName,
                'room_id' => $room->sid,
                'sessionId' => $room->sid,
                'details' => $video,
                'success' => true,
            ];
        } else {
            $resultObject = (object) [
                'success' => false,
            ];
        }
        return $resultObject;
    }

    public function hasMaxParticipants($roomName)
    {
        $participants = $this->twilio->getParticipants($roomName);
        if (count($participants) >= $this->maxParticipants) {
            return true;
        } else {
            return false;
        }
    }

    public function isRoomClosed($roomName)
    {
        $videoRoom = VideoRoom::where('room_name', $roomName)->first();
        Cache::put($videoRoom->room_id, $videoRoom->status);
        if (isset($videoRoom)) {
            if ($videoRoom->status == VideoRoom::VIDEO_COMPLETED) {
               return true;
            }
        } else {
            return false;
        }
    }

    public function disconnectParticipants($roomName)
    {
        $participants = $this->twilio->getParticipants($roomName);

        foreach ($participants as $participant) {
            $this->twilio->disconnectParticipant($roomName, $participant->identity);
        }

        return false;
    }

    public function downloadComposition($compositionSid)
    {
        return $this->twilio->downloadComposition($compositionSid);
    }

    public function storeVideoParticipant($parameter)
    {
        $userId = -1;
        if (isset(auth()->user()->id)) {
            $userId = auth()->user()->id;
        }

        $videoRoomId = VideoRoom::where('room_id', $parameter['roomId'])->first();
        $participant = VideoRoomParticipant::where('participant_sid', $parameter['participantSID'])
            ->where('video_room_id', $videoRoomId->id)
            ->first();

        if (isset($participant->id)) {
            $existingParticipant = VideoRoomParticipant::find($participant->id);
            $existingParticipant->status = 'in-progress';
            $existingParticipant->save();
        } else {
            $participant = new VideoRoomParticipant;
            $participant->video_room_id = $videoRoomId->id;
            $participant->user_id = $userId;
            $participant->participant_sid = $parameter['participantSID'];
            $participant->room_id = $parameter['roomId'];
            $participant->status = 'in-progress';
            $participant->save();
        }
        
        return false;
    }

    public function getTwilioToken()
    {
        return $this->twilio->getToken();
    }

    public function sendSMS($parameter)
    {
        $this->twilio->sendSMS($parameter['mobileNumber']
                , $parameter['message'] 
            );
        
    }
    public function sendMMS($parameter)
    {
        $this->twilio->sendMMS($parameter['mobileNumber']
                , $parameter['message'] 
                , $parameter['image']
            );
        
    }

    public function sendEmail($parameter)
    {
        $emailOnQueue = new EmailOnQueue;
        $emailOnQueue->subject = $parameter['subject'];
        $emailOnQueue->template =   $parameter['template'];
        $emailOnQueue->parameters =   $parameter['parameters'];
        $emailOnQueue->email_address =  $parameter['emailAddress'];
        $emailOnQueue->create_by = auth()->user()->username;
        $emailOnQueue->is_sent = 0;
        $emailOnQueue->sent_date = null;
        $emailOnQueue->save();
    }

    public function createContact($request)
    {
                   
        $contact = Contact::updateOrCreate(
        [
            'reference_id' => $request->referenceId,
        ],
        [   
            'reference_id' => $request->referenceId,
            'full_name' => $request->fullName,    
            'account_name' => $request->accountName,    
            'email_address' => $request->emailAddress,    
            'number_of_monitors' => $request->numberOfMonitors,    
            'headset' => $request->headset,    
            'hardware_type' => $request->hardwareType,    
            'enough_space' => $request->enoughSpace,    
            'outsite_noise_workable' => $request->outsiteNoiseWorkable,    
            'enough_lighting' => $request->enoughLighting,    
            'event_type' => $request->eventType,    
            'room_id' => $request->roomId,    
        ]);
       
        $resultObject = (object) [
            'details' => $contact,
            'success' => true,
        ];
       
        return $resultObject;
    }
    
    public function createRoomEmail($request)
    {
        $roomName = time() . date('YmdHis');
        $mobileNumber = "4845748747";
        $emailAddress = $request->emailAddress;

        if (!$this->twilio->isRoomExists($roomName)) {
            $room = $this->twilio->createRoom($roomName);

            $data = array(
                'link' => url('/') . '/videos/room/join/' . $roomName.'/'.$room->sid,
            );
        
            $emailParameters = array(
                'subject' => 'Stratvideo Onboarding Meeting Link',
                'template' => "mails.pcr",
                'parameters' => serialize($data),
                'emailAddress' => $emailAddress,
            );
            $this->sendEmail($emailParameters);
            
            $video = new VideoRoom;
            $video->mobile_number = $mobileNumber;
            $video->email_address = $emailAddress;
            $video->room_id = $room->sid;
            $video->room_name = $roomName;
            $video->link = url('/') . '/videos/room/join/' . $roomName.'/'.$room->sid;
            $video->status = $room->status;
            $video->user_id = auth()->user()->id;

            //additional optional fields
            if(isset($request->shipped_date))
            {
                $video->shipped_date = $request->shipped_date;    
            }
            if(isset($request->shipped_time))
            {
                $video->shipped_time = $request->shipped_time;    
            }
            if(isset($request->shipped_by))
            {
                $video->shipped_by = $request->shipped_by;    
            }
            if(isset($request->shipping_label_company))
            {
                $video->shipping_label_company = $request->shipping_label_company;    
            }
            if(isset($request->shipping_company))
            {
                $video->shipping_company = $request->shipping_company;    
            }
            $video->save();
            Cache::put($room->sid, $room->status);
            \Log::debug("created new room: " . $roomName);
        
            $resultObject = (object) [
                'roomName' => $roomName,
                'room_id' => $room->sid,
                'sessionId' => $room->sid,
                'details' => $video,
                'success' => true,
            ];
        } else {
            $resultObject = (object) [
                'success' => false,
            ];
        }
        return $resultObject;
    }
}
