<?php

namespace App\Http\Controllers\Admin;

use App\Contracts\Admin\UserActionService;
use App\Models\Role;
use App\Models\RoleUser;
use App\Models\User;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

use DB;
class UserController extends Controller
{

    protected $userActionService;
    
    public function __construct(UserActionService $userActionService)
    {
        $this->middleware('auth');
        $this->middleware('access:users,view|edit|create|delete')->only('index');
        $this->middleware('access:users,view')->only('show');
        $this->middleware('access:users,edit')->only('edit');
        $this->middleware('access:users,delete')->only('delete');
        $this->middleware('access:users,create')->only('create');
        $this->userActionService = $userActionService;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('admin.users.index');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function getData(Request $request)
    {

        $resultObject = $this->userActionService
            ->listUsers();
        return response()->json($resultObject->users);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::where('status','A')->get();
        return view('admin.users.create',compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try
        {
            $this->validate($request,[
                    'first_name' => 'required|string|max:255',
                    'last_name' => 'required|string|max:255',
                    'email' => 'nullable|string|email|max:255|unique:users,email,D,status',
                    'password' => 'required|string|min:6|confirmed',
                    'mobile_number' => 'nullable|string|max:255|unique:users,mobile_number,D,status',
                    'access' => 'required',
            ]);
            $user = new User;
            $user->first_name = $request->input('first_name');
            $user->last_name = $request->input('last_name');
            $user->email = $request->input('email');
            $user->password = bcrypt($request->input('password'));
            $user->mobile_number = $request->input('mobile_number');
            $user->status = $request->input('status') ?? 'A';
            $user->created_at = date('Y-m-d H:i:s');
            $user->create_by = auth()->user()->username;
            $user->save();
            $id = $user->id;
            
            RoleUser::where('user_id', $id)->delete();
            $accesses = substr($request->access, 0, strlen($request->access) - 1);
            $accesses = explode(",", $accesses);
            foreach($accesses as $access) 
            {      
                $roleUser = new RoleUser;
                $roleUser->user_id = $id;
                $roleUser->role_id = $access;
                $roleUser->timestamps = false;
                $roleUser->save(); 
            }
            
            $data = array(
                'first_name' => $request->input('first_name'),
                'last_name' => $request->input('last_name'),
                'username' => $request->input('email'),
                'password' => $request->input('password'),
            );
            $email_params = array(
                'subject' => 'Account Creation',
                'template' => "mails.account_creation",
                'parameters' => serialize($data),
                'email_address' => $request->input('email'),
            );
            $this->userActionService->createEmailOnQueue($email_params);
            return redirect('/admin/users')->with(['success' => 'User added']);
        } catch (Throwable $e) {
            return redirect('/admin/users')->with(['failed' => $e]);     
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = User::find($id);
        $roles = Role::where('status','A')->get();
        $roleUsers = RoleUser::where('user_id',$id)->get();
        $roleUserAccess="";
        foreach($roleUsers as $roleUser)
        {
            $roleUserAccess.= $roleUser->role_id.",";   
        }
        if(strlen($roleUserAccess) >0){
            $roleUserAccess = substr($roleUserAccess, 0, strlen($roleUserAccess) - 1);
        }
        $roleUserAccess = explode(",",$roleUserAccess);
        return view('admin.users.show',compact('user','roles','roleUserAccess'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = User::find($id);
        $roles = Role::where('status','A')->get();
        $roleUsers = RoleUser::where('user_id',$id)->get();
        $roleUserAccess="";
        foreach($roleUsers as $roleUser)
        {
            $roleUserAccess.= $roleUser->role_id.",";   
        }
        if(strlen($roleUserAccess) >0){
            $roleUserAccess = substr($roleUserAccess, 0, strlen($roleUserAccess) - 1);
        }
        $roleUserAccess = explode(",",$roleUserAccess);
        return view('admin.users.edit',compact('user','roles','roleUserAccess'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try
        {
            $this->validate($request,[
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'nullable|string|email|max:255|unique:users,email,'.$id,
                'mobile_number' => 'nullable|string|max:255|unique:users,mobile_number,'.$id,
                'access' => 'required',
            ]);

            $user = User::find($id);
            $user->first_name = $request->input('first_name');
            $user->last_name = $request->input('last_name');
            $user->email = $request->input('email');
            $user->mobile_number = $request->input('mobile_number');
            $user->status = $request->input('status') ?? 'A';
            $user->created_at = date('Y-m-d H:i:s');
            $user->create_by = auth()->user()->username;
            $user->save();

            RoleUser::where('user_id', $id)->delete();
            $accesses = substr($request->access, 0, strlen($request->access) - 1);
            $accesses = explode(",", $accesses);
            foreach($accesses as $access) 
            {      
                $roleUser = new RoleUser;
                $roleUser->user_id = $id;
                $roleUser->role_id = $access;
                $roleUser->timestamps = false;
                $roleUser->save(); 
            }
            return redirect('/admin/users')->with(['success' => 'User was successfully updated']);
        } catch (Throwable $e) {
            return redirect('/admin/users')->with(['failed' => $e]);     
        }

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete($id)
    {
        $user = User::find($id);
        $user->status = 'D';
        $user->save();

        return redirect('/admin/users')->with(['success' => 'User was successfully deleted']);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reset($id)
    {
        $new_password = env('DEFAULT_PASSWORD','rvm2022!');
        $user = User::find($id);
        $user->password = bcrypt($new_password);
        $user->save();

        $data = array(
            'first_name' => $user->first_name,
            'last_name' => $user->last_name,
            'password' => $new_password,
        );

        $email_params = array(
            'subject' => 'Account Reset Password',
            'template' => "mails.reset_password",
            'parameters' => serialize($data),
            'email_address' => $user->email,
        );
        $this->userActionService->createEmailOnQueue($email_params);
        return redirect('/admin/users')->with(['success' => 'Successfult reset password']);
    }

        /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function profile()
    {
        $user = User::find(auth()->user()->id);
        return view('admin.users.profile',compact('user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateProfile(Request $request)
    {
        try
        {
            $id = auth()->user()->id;
            $this->validate($request,[
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'nullable|string|email|max:255|unique:users,email,'.$id,
                'mobile_number' => 'nullable|string|max:255|unique:users,mobile_number,'.$id,
                'job_title' => 'nullable|string|max:255',
                'education' => 'nullable|string|max:255',
                'location' => 'nullable|string|max:255',
            ]);

            $user = User::find($id);
            $user->first_name = $request->input('first_name');
            $user->last_name = $request->input('last_name');
            $user->email = $request->input('email');
            $user->mobile_number = $request->input('mobile_number');
            $user->job_title = $request->input('job_title');
            $user->skills = $request->input('skills');
            $user->location = $request->input('location');
            $user->experience = $request->input('experience');
            $user->education = $request->input('education');
            $user->updated_at = date('Y-m-d H:i:s');
            $user->update_by = auth()->user()->username;
            $user->save();
            return redirect('/admin/users/profile')->with(['success' => 'Profile was successfully updated']);
        } catch (Throwable $e) {
            return redirect('/admin/users/profile')->with(['failed' => $e]);     
        }

    }

    function changePassword(){
		$userId = auth()->user()->id;
		return view('admin.users.change',compact('userId'));	
	}

    public function updatePassword(Request $request)
    {
        try
        {
            $id = auth()->user()->id;
            $this->validate($request,[
                'currentPassword' => 'required|string|max:255',
                'password' => 'required|confirmed|string|max:255',
            ]);

            $user = User::find($id);
            if(Hash::check($request->input('currentPassword'), auth()->user()->password))
            {
                $user = User::find($id);
                $user->password = bcrypt($request->input('password'));
                $user->save();
                return redirect('/admin/users/changePassword')->with(['success' => 'Password was successfully updated']);
            } else {
                return redirect('/admin/users/changePassword')->with(['failed' => 'Current password does not match with our record']);   
            }
        } catch (Throwable $e) {
            return redirect('/admin/users/changePassword')->with(['failed' => $e]);     
        }

    }
}
