<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Twilio\Jwt\AccessToken;
use Twilio\Jwt\Grants\VideoGrant;
use Twilio\Rest\Client;
use Twilio\Exceptions\TwilioException;

use Log;

class Twilio extends Model
{
    protected $sid;
    protected $token;
    protected $key;
    protected $secret;
    protected $maxParticipants;
    protected $twilioNumber;
    protected $client;

    public function __construct()
    {
        $this->sid = config('services.twilio.sid');
        $this->token = config('services.twilio.token');
        $this->key = config('services.twilio.key');
        $this->secret = config('services.twilio.secret');
        $this->maxParticipants = config('services.twilio.maxParticipants');
        $this->twilioNumber = config('services.twilio.mobileNumber');
        $this->client = new Client($this->sid, $this->token);
    }

    /**
     * SMS API
     */
    public function sendSMS($to, $message)
    {

        try {
            $this->client->messages->create(
                $to,
                [
                    "body" => $message,
                    "MessagingServiceSid" => $this->twilioNumber,
                ]
            );
            $resultObject = (object) [
                'success' => true,
                'message' => 'Message sent to ' . $to,
            ];
        } catch (TwilioException $e) {
            $resultObject = (object) [
                'success' => false,
                'message' => 'Could not send SMS notification.  Twilio replied with: ' . $e,
            ];
        }
        Log::info($resultObject->message);
        return $resultObject;
    }

    public function sendMMS($to, $message, $image)
    {

        try {
            $this->client->messages->create(
                $to,
                [
                    "body" => $message,
                    "MessagingServiceSid" => $this->twilioNumber,
                    "mediaUrl" => [$image]
                ]
            );
            $resultObject = (object) [
                'success' => true,
                'message' => 'Message sent to ' . $to,
            ];
        } catch (TwilioException $e) {
            $resultObject = (object) [
                'success' => false,
                'message' => 'Could not send MMS notification.  Twilio replied with: ' . $e,
            ];
        }
        Log::info($resultObject->message);
        return $resultObject;
    }

    /**
     * Video API 
     */

    public function isRoomExists($roomName)
    {
        
        if(empty($this->client->video->rooms->read(['uniqueName' => $roomName])))
        {
            return false;   
        } else {
            return true;
        }  
    }

    public function createRoom($roomName)
    {
        $room = $this->client->video->rooms->create([
            'uniqueName' => $roomName,
            'type' => 'group',
            'recordParticipantsOnConnect' => true,
        ]);

        return $room;
    }

    public function getToken()
    {
        $token = new AccessToken($this->sid, $this->key, $this->secret, 3600, rand(3, 1000));
        return $token;
    }

    public function getParticipants($roomName)
    {
        return $this->client->video->rooms($roomName)
        ->participants->read(array("status" => "connected"));

    }

    public function disconnectParticipant($roomName, $participantIdentity)
    {
        return $this->client->video->rooms($roomName)
        ->participants($participantIdentity)->update(array("status" => "disconnected"));

    }

    public function downloadComposition($compositionSid)
    {
        $uri = "https://video.twilio.com/v1/Compositions/$compositionSid/Media?Ttl=3600";
        $response = $this->client->request("GET", $uri);
        return $response->getContent()["redirect_to"];

    }
}
