<?php

namespace App\Http\Controllers;

use App\Contracts\Video\VideoActionService;

use App\Models\EmailOnQueue;
use App\Models\Message;
use App\Models\Twilio;
use App\Models\VideoRoom;
use App\Models\VideoRoomParticipant;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Redirect;

use Twilio\Jwt\AccessToken;
use Twilio\Jwt\Grants\VideoGrant;
use Twilio\Rest\Client;

use Mail;
class VideoRoomController extends Controller
{
    protected $sid;
    protected $token;
    protected $key;
    protected $secret;

    protected $videoActionService;

    public function __construct(VideoActionService $videoActionService)
    {
        $this->sid = config('services.twilio.sid');
        $this->token = config('services.twilio.token');
        $this->key = config('services.twilio.key');
        $this->secret = config('services.twilio.secret');
        $this->max_participants = config('services.twilio.maxParticipants');
        $this->middleware('access:videos,download')->only('downloadRecording');

        $this->videoActionService = $videoActionService;
    }

    public function index()
    {
        return view('video.index');
    }

    public function getData(Request $request)
    {
        $resultObject = $this->videoActionService
                        ->getVideos();
        return response()->json($resultObject->videos);
    }

    public function room()
    {
        return view('video.room');
    }

    public function createRoom(Request $request)
    {
        $video = $this->videoActionService->createRoom($request); 
        if ($video->success)
        {
            return redirect()->action('VideoRoomController@joinRoom', [
                'roomName' => $video->roomName, 'roomId' => $video->room_id,
            ]);
        } else {
            return redirect('/videos')->with(['failed' => 'Video Room already exists']);     
        }
    }

    public function joinRoom($roomName, $roomId)
    {

        //Validate if room is already completed/closed
        if($this->videoActionService->isRoomClosed($roomName))
        {
            if (isset(auth()->user()->id)) {
                return redirect('/videos')->with(['failed' => 'Video Room already closed']);
            } else {
                return redirect('/videos/close')->with(['failed' => 'Video Room already closed']);
            }
        }

        //Validate max participants
        if($this->videoActionService->hasMaxParticipants($roomName))
        {
            if (isset(auth()->user()->id)) {
                return redirect('/')->with(['failed' => 'Reached maximum number of participants for this room.']);
            } else {
                return redirect('/videos/max')->with(['failed' => 'Max participants already .reached']);
            }
        }
        
        $token = $this->videoActionService->getTwilioToken();

        $videoGrant = new VideoGrant();
        $videoGrant->setRoom($roomName);

        $token->addGrant($videoGrant);
        return view('video.session', ['accessToken' => $token->toJWT(), 'roomName' => $roomName, 'roomId' => $roomId]);
    }

    public function completeRoom($roomId)
    {
        $mmsImage = env('TWILIO_MMS_IMAGE');
        
        $video = VideoRoom::where('room_id', $roomId)->first();
        $video->status = 'completed';
        $video->save();
        
        $parameter = array(
            'mobileNumber' => $video->mobile_number,
            'message'   => Message::VIDEO_CLOSING_REMARK,
            'image' => $mmsImage
        );
        
        $this->videoActionService->sendMMS($parameter);
        Cache::put($roomId, 'completed');

        $this->videoActionService->disconnectParticipants($video->room_name);

        if (isset(auth()->user()->id)) {
            return redirect()->action('VideoRoomController@index');
        } else {
            return redirect('/videos/close')->with(['failed' => 'Video Room already closed']);
        }
    }

    public function checkRoomSession()
    {
        $status = Cache::get($_GET['roomId'], '');
        if (isset(auth()->user()->id)) {
            $status = 'in-progress'; //avoid ending session on agent screen
        }
        return response()->json($status);
    }

    public function downloadRecording($compositionSid)
    {
        $mediaLocation =  $this->videoActionService->downloadComposition($compositionSid);
        return Redirect::to($mediaLocation);
    }

    public function storeVideoParticipant()
    {
        $this->videoActionService->storeVideoParticipant($_GET);
    }

    public function hideVideo()
    {
        $participant = VideoRoomParticipant::where('participant_sid', $_GET['participantSID'])
            ->where('room_id', $_GET['roomId'])
            ->first();

        if (isset($participant)) {
            $participant->status = $_GET['participantStatus'];
            $participant->save();
        }

        return response()->json($participant);
    }

    public function checkVideoStatus()
    {
        $participants = VideoRoomParticipant::where('room_id', $_GET['roomId'])
            ->get();

        $result = array();
        foreach ($participants as $participant) {
            $result[] = array(
                'status' => $participant->status,
                'participantSID' => $participant->participant_sid,
            );
        }

        $resultObject = (object) [
            'videos' => $result,
        ];
        return response()->json($resultObject->videos);
    }

    
    public function smsTest()
    {
        
    }
}
