<?php

namespace App\Http\Controllers\Admin;

use App\Contracts\Admin\PermissionActionService;
use App\Http\Controllers\Controller;
use App\Models\Permission;
use App\Models\PermissionGroup;
use Illuminate\Http\Request;

class PermissionController extends Controller
{

    protected $permissionActionService;
    
    public function __construct(PermissionActionService $permissionActionService)
    {
        $this->middleware('access:permissions,view|edit|create|delete')->only('index');
        $this->middleware('access:permissions,view')->only('show');
        $this->middleware('access:permissions,edit')->only('edit');
        $this->middleware('access:permissions,delete')->only('delete');
        $this->middleware('access:permissions,create')->only('create');
        $this->permissionActionService = $permissionActionService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('admin.permissions.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $categories = PermissionGroup::isActive()->get();
        return view('admin.permissions.create',compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try
        {
            $this->validate($request,[
                    'permission' => 'required|string|max:255',
                    'url' => 'required|string|max:255',
                    'description' => 'required|string|max:255',       
            ]);
            $permission = new Permission;
            $permission->permission = $request->input('permission');
            $permission->url = $request->input('url');
            $permission->description = $request->input('description');
            $permission->permission_group_id = $request->input('category');
            $permission->supported_right = $request->input('access');
            $permission->status = $request->input('status') ?? 'A';
            $permission->created_at = date('Y-m-d H:i:s');
            $permission->create_by = auth()->user()->username;
            $permission->save();
           
            return redirect('/admin/permissions')->with(['success' => 'Permission added']);
        } catch (Throwable $e) {
            return redirect('/admin/permissions')->with(['failed' => $e]);     
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $permission = Permission::find($id);
        $categories = PermissionGroup::isActive()->get();
        return view('admin.permissions.show',compact('permission','categories'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $permission = Permission::find($id);
        $categories = PermissionGroup::isActive()->get();
        return view('admin.permissions.edit',compact('permission','categories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try
        {
            $this->validate($request,[
                    'permission' => 'required|string|max:255',
                    'url' => 'required|string|max:255',
                    'description' => 'required|string|max:255',       
            ]);
            $permission = Permission::find($id);
            $permission->permission = $request->input('permission');
            $permission->url = $request->input('url');
            $permission->description = $request->input('description');
            $permission->permission_group_id = $request->input('category');
            $permission->supported_right = $request->input('access');
            $permission->status = $request->input('status') ?? 'A';
            $permission->updated_at = date('Y-m-d H:i:s');
            $permission->update_by = auth()->user()->username;
            $permission->save();
           
            return redirect('/admin/permissions')->with(['success' => 'Permission updated']);
        } catch (Throwable $e) {
            return redirect('/admin/permissions')->with(['failed' => $e]);     
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete($id)
    {
        $permission = Permission::find($id);
        $permission->status = 'D';
        $permission->save();

        return redirect('/admin/permissions')->with(['success' => 'Permission was successfully deleted']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function getData(Request $request)
    {

        $resultObject = $this->permissionActionService
            ->listPermissions();
        return response()->json($resultObject->permissions);
    }

    /**
     * Display supported access rights based on category
     */
    public function getSupportedRights(Request $request)
    {

        $resultObject = $this->permissionActionService
            ->getSupportedRights($request->categoryId);
        return response()->json($resultObject->supportedRights);
    }
}
